<?php

namespace backend\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * Class Admin
 * @package backend\models
 *
 * @property string $login
 * @property string $password_hash
 * @property integer $role
 *
 */
class Admin extends ActiveRecord implements IdentityInterface
{

    const ROLE_ADMIN = 0;
    const ROLE_MANAGER = 10;
    const ROLE_SUPER_ADMIN = 20;

    public $new_password;
    public $confirm_password;

    /**
     * @inheritdoc
     */
    public static function tableName()
    {
        return 'admin';
    }

    /**
     * @inheritdoc
     */
    public function attributeLabels()
    {
        return [
            'login' => 'Login',
            'password_hash' => 'Password',
            'new_password' => 'Password',
            'confirm_password' => 'Confirm password',
            'role' => 'Role',
            'groups' => 'Access to groups',
            'status' => 'Status',
        ];
    }

    /**
     * @return array the validation rules.
     */
    public function rules()
    {
        return [
            [['login'], 'required'],
            [['login'], 'unique'],
            [['role', 'status'], 'integer'],
            [['groups'], 'string'],
            [['login', 'password_hash', 'new_password', 'confirm_password',], 'string'],
            ['confirm_password', 'compare', 'compareAttribute' => 'new_password', 'message' => "Passwords not match."],
            [['password_hash'], 'string', 'max' => 500],
            ['role', 'in', 'range' => [static::ROLE_ADMIN, static::ROLE_MANAGER, static::ROLE_SUPER_ADMIN]],
            [['new_password', 'confirm_password'], 'string', 'min' => 5],
        ];
    }
    
    
    /**
     * Finds an identity by the given ID.
     *
     * @param string|int $id the ID to be looked for
     * @return IdentityInterface|null the identity object that matches the given ID.
     */
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    /**
     * Finds an identity by the given login.
     *
     * @param $login
     * @return array|null|ActiveRecord
     */
    public static function findByLogin($login)
    {
        return static::find()->where(['login' => $login])->one();
    }

    /**
     * Stub
     * @inheritdoc
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    /**
     * @return int|string current user ID
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Stub
     * @inheritdoc
     */
    public function getAuthKey()
    {
        return null;
    }

    /**
     * Stub
     * @inheritdoc
     */
    public function validateAuthKey($authKey)
    {
        return false;
    }

    /**
     * @inheritdoc
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    /**
     * Generates password hash from password and sets it to the model
     *
     * @param string $password
     */
    public function setPassword($password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    public function getAdminSettingValue($name)
    {
        $s = (array)json_decode(Yii::$app->user->identity->settings);

        if(isset($s[$name])){
            return $s[$name];
        }else{
            return false;
        }
    }

    public function setAdminSettingValue($name, $value)
    {
        $a = static::find()->where(['id' => Yii::$app->user->identity->id])->one();
        $s = (array)json_decode($a->settings);
        $s[$name] = $value;
        $a->settings = json_encode($s);
        if($a->save(false)){
            return true;
        }else{
            return false;
        }
    }

}